#include "Gate_INPUT.h"

Gate_INPUT::Gate_INPUT(int gateID, gType type, int numInputs, int gateLevel)
    : Gate_BASE(gateID, type, numInputs, gateLevel)
{
    // Override base gate to limit to one output point only
    this->xSize = BASE_GATE_SIZE_X + ADDITONAL_INPUTS;
    this->ySize = BASE_GATE_SIZE_Y + ADDITONAL_INPUTS;

    outputPoint = QPointF(this->xSize, this->ySize / 2);
    gateOutputPoint = QPointF(this->scenePos().x() + this->xSize, this->scenePos().y() + this->ySize / 2);

    textRect = QRectF(BORDER_OFFSET, BORDER_OFFSET, xSize - BORDER_OFFSET * 3, ySize - BORDER_OFFSET * 2);

    createActions();
}

void Gate_INPUT::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(widget);
    Q_UNUSED(option);

#ifdef _DEBUG
    painter->save();
    painter->setPen((auxSelected) ? debugSelectedPen : debugPen);
    painter->setBrush(debugBrush);
    painter->drawRect(0, 0, xSize, ySize);
    painter->drawRect(textRect);
    if (numInputs != 0) {
        painter->setPen(debugErrorPen);
        painter->drawLine(0, 0, xSize, ySize);
        painter->drawLine(xSize, 0, 0, ySize);
    }
    painter->restore();
#endif

    if (auxSelected) painter->setPen(highlightedPen);
    else painter->setPen(defaultPen);
    painter->setBrush(defaultBrush);

    // Draw circles indicating I/O points
    painter->drawEllipse(outputPoint, INPUT_CIRCLE_SIZE, INPUT_CIRCLE_SIZE);

    // Draw gate outline
    QPointF points[5] = {
        QPointF(BORDER_OFFSET, BORDER_OFFSET),
        QPointF(xSize - BORDER_OFFSET * 2, BORDER_OFFSET),
        QPointF(xSize - BORDER_OFFSET, ySize / 2),
        QPointF(xSize - BORDER_OFFSET * 2, ySize - BORDER_OFFSET),
        QPointF(BORDER_OFFSET, ySize - BORDER_OFFSET)
    };
    painter->drawPolygon(points, 5);

    // Draw I/O lines
    painter->drawLine(outputPoint, QPointF(outputPoint.x() - BORDER_OFFSET, outputPoint.y()));

    // Draw text showing gate ID
    painter->setPen(defaultPen);
    painter->setFont(defaultFont);
    painter->drawText(textRect, Qt::AlignCenter, QString::number(gateID));

    // If enqueued, draw circle around gate ID
    if (enqueued) {
        painter->drawEllipse(textRect.center(), ENQUEUED_CIRCLE_WIDTH, ENQUEUED_CIRCLE_WIDTH);
    }
}

void Gate_INPUT::simulateToOutput()
{
    // Do nothing

    // Update connected wire values
    for (int i = 0; i < gateOutputWires.size(); i++) {
        gateOutputWires[i]->setValue(outputValue, outputFaultyValue, false);
    }
}

void Gate_INPUT::promptValue()
{
    // Prompt for the input value
    bool ok;
    QStringList options;
    options << "0" << "1" << "X";
    QString sel = QInputDialog::getItem(0, "Choose Value", "Enter Value:", options, 0, false, &ok);
    if (ok && !sel.isEmpty()) {
        if (sel == "0") {
            outputValue = logicValue_0;
            outputFaultyValue = logicValue_0;
        } else if (sel == "1") {
            outputValue = logicValue_1;
            outputFaultyValue = logicValue_1;
        } else {
            outputValue = logicValue_X;
            outputFaultyValue = logicValue_X;
        }
    }

    // Propogate to connected wires
    for (int i = 0; i < gateOutputWires.size(); i++) {
        gateOutputWires[i]->setValue(outputValue, outputFaultyValue, false);
    }
}

void Gate_INPUT::promptFaultyValue()
{
    // Prompt for the fault value
    bool ok;
    QStringList options;
    options << "0" << "1" << "X";
    QString sel = QInputDialog::getItem(0, "Choose Value", "Enter Value:", options, 0, false, &ok);
    if (ok && !sel.isEmpty()) {
        if (sel == "0") {
            outputValue = logicValue_1;
            outputFaultyValue = logicValue_0;
        } else if (sel == "1") {
            outputValue = logicValue_0;
            outputFaultyValue = logicValue_1;
        } else {
            outputValue = logicValue_X;
            outputFaultyValue = logicValue_X;
        }
    }

    // Propogate to connected wires
    for (int i = 0; i < gateOutputWires.size(); i++) {
        gateOutputWires[i]->setValue(outputValue, outputFaultyValue, false);
    }
}

void Gate_INPUT::contextMenuEvent(QGraphicsSceneContextMenuEvent *event)
{
    QMenu menu;
    menu.addAction(setInputValueAction);
    menu.addAction(setFaultValueAction);
    menu.exec(event->screenPos());
    event->accept();
}

void Gate_INPUT::createActions()
{
    setInputValueAction = new QAction("&Set Value", this);
    connect(setInputValueAction, SIGNAL(triggered()), this, SLOT(promptValue()));

    setFaultValueAction = new QAction("&Inject Fault", this);
    connect(setFaultValueAction, SIGNAL(triggered()), this, SLOT(promptFaultyValue()));
}
